module net.BurtonRadons.dig.platform.button;

private import net.BurtonRadons.dig.platform.control;

/** This widget is a push button with a text or bitmap label.
  Until assigned, the size of it fits its contents plus a
  border.

  The main button type is a text button, which you assign using
  the #caption and #formatCaption methods.  Other methods modify
  this appearance; #bordered puts a darker border around the
  button (commonly used with default buttons), #flat makes the
  button flat instead of 3D if possible, #alignLeft and #alignRight
  modify the horizontal position of the text in
  its space, and #alignTop modifies the vertical position of the
  text.

  These are the supported dispatchers:
  - #onClick: Sent when the button is clicked on.
  - #onDClick: Sent when the button is clicked on again in rapid succession (a double-click); if this is empty, #onClick is notified instead.

  */

class Button : public Control
{
    private import net.BurtonRadons.dig.platform.base;
    private import net.BurtonRadons.dig.platform.font;
    
    Dispatcher onClick; /**< Dispatched when the button is clicked upon. */
    Dispatcher onDClick; /**< Dispatched when the button is double-clicked upon; if this is empty, #onClick is used instead. */

    /** Register with the parent and create the button. */
    this (Control parent)
    {
        super (parent);
        digPlatformStyle |= std.c.windows.windows.WS_VISIBLE | std.c.windows.windows.WS_CHILD | std.c.windows.windows.BS_NOTIFY;
        digPlatformStyle &= ~(std.c.windows.windows.WS_HSCROLL | std.c.windows.windows.WS_VSCROLL);
        digPlatformHWNDCreate (0, "Button", null, digPlatformStyle, (_HANDLE) 0);
    }

    /** Change the textual caption in the button.
      * Text where & is followed by a character is interpreted
      * specially to underline that character, and binds it to have the
      * same action as clicking on the button.
      */

    void caption (char [] text)
    {
        if (digPlatformBoundLetter [0])
            bindings.remove (digPlatformBoundLetter, &onClick);

        digPlatformSetCaption (text);
        recalculate ();

        int offset = std.string.find (text, '&');

        if (offset >= 0)
        {
            digPlatformBoundLetter [0] = text [offset + 1];
            bind (digPlatformBoundLetter, &onClick);
        }
    }

    /** If set, this takes a darker border, indicating it's the default. */
    void bordered (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_DEFPUSHBUTTON, value); }

    /** If set, this button is flat, instead of 3d. */
    void flat (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_FLAT, value); }

    /** If set, the text inside the button is aligned to the left border; otherwise it is centered. */
    void alignLeft (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_LEFT, value); }

    /** If set, the text inside the button is aligned to the right border; otherwise it is centered. */
    void alignRight (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_RIGHT, value); }

    /** If set, the text inside the button is aligned to the top; otherwise it is vertically centered. */
    void alignTop (bit value) { digPlatformSetStyle (std.c.windows.windows.BS_TOP, value); }

    /** Return whether the button is pushed down. */
    bit pushed ()
    {
        int value = SendMessageA (digPlatformHWND, std.c.windows.windows.BM_GETSTATE, 0, 0);

        return (bit) (value & std.c.windows.windows.BST_PUSHED);
    }

    /** Set whether the button is pushed down. */
    void pushed (bit value)
    {
		SendMessageA (digPlatformHWND, std.c.windows.windows.BM_SETSTATE, value, 0);
    }

    override void recalculate ()
    {
        _HGDIOBJ font;
        _DWORD result;

        font = (_HANDLE) SendMessageA (digPlatformHWND, WM_GETFONT, 0, 0);
        SelectObject (digPlatformHDC, font);
        result = GetTabbedTextExtentW (digPlatformHDC, digPlatformCaption, digPlatformCaption.length, 0, null);
        digPlatformSuggestWidth = result & 0xFFFF;
        digPlatformSuggestHeight = result >> 16;
        digPlatformBorder (digPlatformSuggestWidth, digPlatformSuggestHeight);
        digPlatformMoved ();
    }

protected:
/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/

    char [1] digPlatformBoundLetter; /**< The letter that this is bound to, or \0 for none. */
    
    override void digPlatformCommand (int code, int id)
    {
        switch (code)
        {
            case std.c.windows.windows.BN_CLICKED: onClick.notify (); break;
            case std.c.windows.windows.BN_DBLCLK: onClick.notifyOrEmpty (&onDClick); break;
            case std.c.windows.windows.BN_KILLFOCUS: break;
            case std.c.windows.windows.BN_SETFOCUS: break;
            default:
                printf ("not handled: %d %d\n", code, id);
        }
    }

    void digPlatformBorder (inout int width, inout int height)
    {
        width = width + 6;
        height = height + 6;
    }

/+
#endif
+/
};

